<?php
include 'config.php';  // Include database connection and configuration settings

function handleSpeedUp($number, $message, $timestamp) {
    error_log("Received data - Number: $number, Message: $message, Timestamp: $timestamp");

    if (!preg_match('/^([\d,]+) speed up$/i', $message, $matches)) {
        sendWhatsAppMessage($number, "Please format your request as 'ID1,ID2,ID3 speed up'.");
        return;
    }

    $orderIds = $matches[1];
    $orderIdsArray = array_map('trim', explode(',', $orderIds));
    $orderIdsParam = implode(',', $orderIdsArray);
    passToApi($orderIdsParam, 'speed up'); // Sending all IDs at once for initial API handling

    $recentComplaints = [];
    $secondTimeComplaints = [];
    $statusGroups = [];

    foreach ($orderIdsArray as $orderId) {
        $hoursSinceLastComplaint = checkRecentComplaint($orderId);

        if ($hoursSinceLastComplaint !== false && $hoursSinceLastComplaint < 12) {
            $recentComplaints[] = $orderId; // Group recent complaints
            continue; // Skip further processing for this order ID
        } else if ($hoursSinceLastComplaint !== false && $hoursSinceLastComplaint >= 12) {
            $secondTimeComplaints[] = $orderId; // Group second-time complaints
            continue; // Skip further processing for this order ID
        }

        $status = getOrderStatus($orderId);
        if (!$status) {
            sendWhatsAppMessage($number, "Failed to retrieve order status for ID $orderId. Please try again or contact support.");
            continue;
        }

        if (!isset($statusGroups[$status])) {
            $statusGroups[$status] = [];
        }
        $statusGroups[$status][] = $orderId;
    }

    // Process and send messages for grouped recent complaints
    if (!empty($recentComplaints)) {
        $idsString = implode(', ', $recentComplaints);
        sendWhatsAppMessage($number, "Your support request for Order IDs $idsString is already in progress. Please allow some time for resolution.");
    }

    // Handle second-time complaints in bulk
    if (!empty($secondTimeComplaints)) {
        include_once('handle_second_time_complaint.php'); // Make sure this file path is correct
        handleSecondTimeComplaint($conn, $number, $secondTimeComplaints, $message);
    }

    // Sending a single message for each group of order IDs with the same status
    foreach ($statusGroups as $status => $ids) {
        $idsString = implode(', ', $ids);
        $orderPhrase = count($ids) > 1 ? "Order IDs $idsString" : "Order ID $idsString";

        switch ($status) {
            case 'completed':
            case 'canceled':
                sendWhatsAppMessage($number, "$orderPhrase is already marked as '$status' and cannot be expedited.");
                break;
            case 'fail':
            case 'error':
                sendWhatsAppMessage($number, "$orderPhrase is currently marked as '$status'. It has been noted for immediate attention and expedited handling.");
                sendGroupMessage('120363365021394212@g.us', "Immediate review needed: $orderPhrase encountered issues.");
                break;
            default:
                sendWhatsAppMessage($number, "$orderPhrase is currently '$status'. It has been added for speeding up to ensure quicker completion.");
                sendGroupMessage('120363365021394212@g.us', "Expedite processing for $orderPhrase.");
                break;
        }
    }
}

function getOrderStatus($orderId) {
    global $conn;
    $stmt = $conn->prepare("SELECT status FROM orders WHERE id = ?");
    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        return false;
    }
    $stmt->bind_param('i', $orderId);
    if (!$stmt->execute()) {
        error_log("Execute failed: " . $stmt->error);
        $stmt->close();
        return false;
    }
    $stmt->bind_result($status);
    $stmt->fetch();
    $stmt->close();
    return $status;
}

function checkRecentComplaint($orderId) {
    global $conn;
    // Fetch the last two complaints, we will check the timing of the earlier one
    $stmt = $conn->prepare("SELECT created_at FROM complaints WHERE order_id = ? ORDER BY created_at DESC LIMIT 2");
    $stmt->bind_param('i', $orderId);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows > 1) {  // Ensure there is more than one complaint
        $stmt->bind_result($lastComplaintTime);
        $stmt->fetch(); // Skip the most recent complaint
        $stmt->fetch(); // Fetch the next most recent complaint
        $stmt->close();

        $currentTime = new DateTime();
        $lastComplaintDateTime = new DateTime($lastComplaintTime);
        $interval = $currentTime->diff($lastComplaintDateTime);
        $hours = $interval->h + ($interval->days * 24);  // Calculate total hours

        error_log("Order ID $orderId: Last relevant complaint was $hours hours ago, Last complaint time: $lastComplaintTime, Current time: " . $currentTime->format('Y-m-d H:i:s'));

        return $hours;
    } else {
        $stmt->close();
        return false;  // No previous complaints or only one complaint found
    }
}



function sendWhatsAppMessage($number, $message) {
    // Ensure number is in the correct format without '@c.us'
    $formattedNumber = str_replace('@c.us', '', $number); // Strip out the suffix if present

    $url = 'http://147.93.136.172:3000/send';
    $data = json_encode(["number" => $formattedNumber, "message" => $message]);
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    
    if ($err = curl_error($ch)) {
        error_log("Curl error in sendWhatsAppMessage to $formattedNumber: $err");
    } else {
        $responseData = json_decode($response, true);
        if (isset($responseData['error'])) {
            error_log("Failed to send message to $formattedNumber: " . json_encode($responseData));
        } else {
            error_log("Success in sending message to $formattedNumber: " . $response);
        }
    }
    curl_close($ch);
}



function sendGroupMessage($groupId, $message) {
    // Ensure group ID is formatted correctly with '@g.us'
    $url = 'http://147.93.136.172:3000/send-group';
    $data = json_encode(["groupId" => $groupId, "message" => $message]); // Directly using the group ID
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    if ($err = curl_error($ch)) {
        error_log("Curl error in sendGroupMessage: $err");
    } else {
        $responseData = json_decode($response, true);
        if (isset($responseData['error'])) {
            error_log("Failed to send group message to $groupId: $response");
        } else {
            error_log("Success in sending message to $groupId: " . $response);
        }
    }
    curl_close($ch);
}


function passToApi($orderIds, $complaintType) {
    $url = 'http://gama.yaseenteam.com/api.php';
    $postData = ['order_ids' => $orderIds, 'complaint_type' => $complaintType];
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    if ($err = curl_error($ch)) {
        error_log("Curl error in passToApi: $err");
    } else {
        error_log("API response: " . $response);
    }
    curl_close($ch);
}
?>
