<?php
include 'config.php'; // Ensure this file includes database connection details

/**
 * Fetches the WhatsApp group ID for a provider by their name.
 *
 * @param mysqli $conn The database connection object.
 * @param string $providerName The name of the provider.
 * @return string|null The WhatsApp group ID or null if not found.
 */
function getWhatsAppGroupIdByProviderName($conn, $providerName) {
    $sql = "SELECT group_id FROM providers WHERE provider_name = ?";
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        error_log("MySQL prepare error: " . $conn->error);
        return null;
    }
    $stmt->bind_param("s", $providerName);
    $stmt->execute();
    $stmt->bind_result($groupId);
    if ($stmt->fetch()) {
        $stmt->close();
        return $groupId;
    } else {
        $stmt->close();
        error_log("No WhatsApp group ID found for provider: " . $providerName);
        return null;
    }
}



/**
 * Retrieves the provider name associated with a given order ID.
 *
 * @param mysqli $conn The database connection object.
 * @param int $orderId The ID of the order.
 * @return string|null The provider name or null if not found.
 */

function getProviderDetailsByOrderId($conn, $orderId) {
    $sql = "SELECT provider, external_id FROM orders WHERE id = ?";
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        error_log("MySQL prepare error: " . $conn->error);
        return null;
    }
    $stmt->bind_param("i", $orderId);
    $stmt->execute();
    $stmt->bind_result($providerName, $externalId);
    if ($stmt->fetch()) {
        $stmt->close();
        return ['providerName' => $providerName, 'externalId' => $externalId];
    } else {
        $stmt->close();
        error_log("No provider or external ID found for order ID: " . $orderId);
        return null;
    }
}

?>
