<?php
include 'config.php';  
require_once 'providerMessaging.php';  

function handleNotStarted($conn, $number, $message, $timestamp) {
    error_log("Received data - Number: $number, Message: $message, Timestamp: $timestamp");

if (!preg_match('/^([\d,]+)\s*(-\s*)?start$/i', $message, $matches)) {
    error_log("Received improperly formatted speed up request from $number: $message");
    return; 
}


    $orderIds = $matches[1];
    $orderIdsArray = array_map('trim', explode(',', $orderIds));
    $orderIdsParam = implode(',', $orderIdsArray);
    passToApi($orderIdsParam, 'not started'); 

    $recentComplaints = [];
    $secondTimeComplaints = [];
    $statusGroups = [];
    $providerMessages = [];  
    $failedToRetrieveDetails = [];

    foreach ($orderIdsArray as $orderId) {
        $hoursSinceLastComplaint = checkRecentComplaint($orderId);

        if ($hoursSinceLastComplaint !== false && $hoursSinceLastComplaint < 3) {
            $recentComplaints[] = $orderId; 
            continue; 
        } else if ($hoursSinceLastComplaint !== false && $hoursSinceLastComplaint >= 3) {
            $secondTimeComplaints[] = $orderId; 
            continue; 
        }

        $status = getOrderStatus($orderId);
        $details = getProviderDetailsByOrderId($conn, $orderId); 
        if (!$status || !$details) {
            $failedToRetrieveDetails[] = $orderId;  
            continue;
        }

        $providerName = $details['providerName'];
        $externalId = $details['externalId'];
        $groupId = getWhatsAppGroupIdByProviderName($conn, $providerName);
        $providerKey = $providerName . '|' . $status;

        if (!isset($statusGroups[$status])) {
            $statusGroups[$status] = [];
        }
        $statusGroups[$status][] = $orderId;

        if (!isset($providerMessages[$providerKey])) {
            $providerMessages[$providerKey] = [
                'orderIds' => [],  
                'externalIds' => [],
                'groupId' => $groupId ?: '120363365021394212@g.us',  
                'message' => determineProviderMessageByStatus($status)
            ];
        }
        $providerMessages[$providerKey]['externalIds'][] = $externalId;
        $providerMessages[$providerKey]['orderIds'][] = $orderId;  
    }

    
    if (!empty($failedToRetrieveDetails)) {
        $idsString = implode(', ', $failedToRetrieveDetails);
        sendWhatsAppMessage($number, "Failed to retrieve order details for IDs: $idsString. Please try again or contact support.");
    }
    
    
    if (!empty($recentComplaints)) {
        $idsString = implode(', ', $recentComplaints);
        sendWhatsAppMessage($number, "Your support request start for Order IDs $idsString is already in progress. Please allow some time for resolution.");
    }

    
    if (!empty($secondTimeComplaints)) {
        include_once('handle_second_time_complaint_not.php'); 
        handleSecondTimeComplaintNot($conn, $number, $secondTimeComplaints, $message);
    }

    
    foreach ($statusGroups as $status => $ids) {
        $idsString = implode(', ', $ids);
        $orderPhrase = count($ids) > 1 ? "Order IDs $idsString" : "Order ID $idsString";
        $userMessage = determineUserMessageByStatus($status, $orderPhrase);
        sendWhatsAppMessage($number, $userMessage);
    }


define('ERROR_GROUP_ID', '120363376465503407@g.us');  
define('SPECIAL_GROUP_ID', '120363373234328549@g.us');  


foreach ($providerMessages as $key => $info) {
    list($providerName, $status) = explode('|', $key);

    
    $providerMessageContent = determineProviderMessageByStatus($status);
    
    
    if ($providerMessageContent === null) {
        error_log("No action needed or message skipped for provider: " . $providerName . " with status: " . $status);
        continue;
    }

    if ($status === 'fail' || $status === 'error') {
        
        $idsString = implode(', ', $info['orderIds']);  
        $providerMessage = $providerMessageContent . " : " . $idsString . ".";
        sendGroupMessage(ERROR_GROUP_ID, $providerMessage);
    } else {
        
        $externalIdsString = implode(', ', $info['externalIds']);
        $providerMessage = $providerMessageContent . $externalIdsString . " Urgent Start";

    
    if ($info['groupId'] === SPECIAL_GROUP_ID) {
    $providerMessage = "Provider " . $providerName . " : " . $providerMessage;
    }

    
     if ($info['groupId']) {
        sendGroupMessage($info['groupId'], $providerMessage);
    } else {
        error_log("No group ID found for provider: " + $providerName);
      }
    }
  }
}


function determineUserMessageByStatus($status, $orderPhrase) {
    switch ($status) {
        case 'completed':
        case 'canceled':
        case 'partial':
            return "$orderPhrase is already marked as $status.";
        case 'fail':
        case 'error':
            return "Your $orderPhrase added for start queue.";
        default:
            return "Your $orderPhrase will start soon.";
    }
}

function determineProviderMessageByStatus($status) {
    switch ($status) {
        case 'completed':
        case 'canceled':
        case 'partial':
            return null;
        case 'fail':
            return "Start For Fail";
        case 'error':
            return "Start For Error";
        default:
            return "";
    }
}

function getOrderStatus($orderId) {
    global $conn;
    $stmt = $conn->prepare("SELECT status FROM orders WHERE id = ?");
    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        return false;
    }
    $stmt->bind_param('i', $orderId);
    if (!$stmt->execute()) {
        error_log("Execute failed: " . $stmt->error);
        $stmt->close();
        return false;
    }
    $stmt->bind_result($status);
    $stmt->fetch();
    $stmt->close();
    return $status;
}

function checkRecentComplaint($orderId) {
    global $conn;
    
    $stmt = $conn->prepare("SELECT created_at FROM complaints WHERE order_id = ? ORDER BY created_at DESC LIMIT 2");
    $stmt->bind_param('i', $orderId);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows > 1) {  
        $stmt->bind_result($lastComplaintTime);
        $stmt->fetch(); 
        $stmt->fetch(); 
        $stmt->close();

        $currentTime = new DateTime();
        $lastComplaintDateTime = new DateTime($lastComplaintTime);
        $interval = $currentTime->diff($lastComplaintDateTime);
        $hours = $interval->h + ($interval->days * 24);  

        error_log("Order ID $orderId: Last relevant complaint was $hours hours ago, Last complaint time: $lastComplaintTime, Current time: " . $currentTime->format('Y-m-d H:i:s'));

        return $hours;
    } else {
        $stmt->close();
        return false;  
    }
}



function sendWhatsAppMessage($number, $message) {
    
    $formattedNumber = str_replace('@c.us', '', $number); 

    $url = 'http://apexfarhan.com:3001/send-group';
    $data = json_encode(["groupId" => $formattedNumber, "message" => $message]);
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    
    if ($err = curl_error($ch)) {
        error_log("Curl error in sendWhatsAppMessage to $formattedNumber: $err");
    } else {
        $responseData = json_decode($response, true);
        if (isset($responseData['error'])) {
            error_log("Failed to send message for start to $formattedNumber: " . json_encode($responseData));
        } else {
            error_log("Success user message for start to $formattedNumber: " . $response);
        }
    }
    curl_close($ch);
}



function sendGroupMessage($groupId, $message) {
    
    $url = 'http://apexfarhan.com:3001/send-group';
    $data = json_encode(["groupId" => $groupId, "message" => $message]); 
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    if ($err = curl_error($ch)) {
        error_log("Curl error in sendGroupMessage: $err");
    } else {
        $responseData = json_decode($response, true);
        if (isset($responseData['error'])) {
            error_log("Failed to send start provider message to $groupId: $response");
        } else {
            error_log("Success - start provider message to $groupId: " . $response);
        }
    }
    curl_close($ch);
}


function passToApi($orderIds, $complaintType) {
    $url = 'http://apexfarhan.com/api.php';
    $postData = ['order_ids' => $orderIds, 'complaint_type' => $complaintType];
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    if ($err = curl_error($ch)) {
        error_log("Curl error in passToApi: $err");
    } else {
        error_log("API response: " . $response);
    }
    curl_close($ch);
}
?>
